﻿#include <cassert>
#include <cstdlib>
#include <stdexcept>
#include <iostream>
#include <thread>
#include <boost/asio.hpp>
#include "digitalcurling3/digitalcurling3.hpp"

#include <time.h>

namespace dc = digitalcurling3;

namespace
{

    /// \brief ティーの位置
    constexpr dc::Vector2 kTee(
        dc::coordinate::GetCenterLineX(dc::coordinate::Id::kShot0),
        dc::coordinate::GetTeeLineY(true, dc::coordinate::Id::kShot0));

    /// \brief GameState::Stones のインデックス．
    struct StoneIndex
    {
        size_t team;
        size_t stone;
    };

    /// \brief ティー(ハウスの中心)からの距離でストーンをソートする．
    ///
    /// \param result 出力先パラメータ．ソート結果のインデックスが格納される．
    ///
    /// \param stones ストーンの配置
    void SortStones(std::array<StoneIndex, 16> &result, dc::GameState::Stones const &stones)
    {
        for (size_t i = 0; i < 16; ++i)
        {
            result[i].team = i / 8;
            result[i].stone = i % 8;
        }

        for (size_t i = 1; i < 16; ++i)
        {
            for (size_t j = i; j > 0; --j)
            {
                auto const &stone_a = stones[result[j - 1].team][result[j - 1].stone];
                auto const &stone_b = stones[result[j].team][result[j].stone];

                if (!stone_a || (stone_b && (stone_a->position - kTee).Length() > (stone_b->position - kTee).Length()))
                {
                    std::swap(result[j - 1], result[j]);
                }
            }
        }
    }

    /// \brief ストーンがハウス内にあるかを調べる．
    ///
    /// \return ストーンがハウス内にある場合 true ，そうでない場合 false
    bool IsInHouse(std::optional<dc::Transform> const &stone)
    {
        if (!stone)
            return false; // 盤面上にストーンが存在しない場合は false
        return (stone->position - kTee).Length() < dc::coordinate::kHouseRadius + dc::ISimulator::kStoneRadius;
    }

    /// \brief シミュレータFCV1において，指定地点を指定速度で通過するショットの初速を逆算します．
    ///
    /// 使用上の注意：
    /// - この関数はシミュレータFCV1に特化した関数です．他のシミュレータには対応していません．
    ///   したがって，この関数を使用した場合必然的にシミュレータFCV1に特化した思考エンジンになります．
    ///  （余談：複数のシミュレータに対応したショット速度を逆算する関数をライブラリから提供しさえすれば，
    ///   その関数を使用することで思考エンジンが特定のシミュレータに特化するといったことは無くなります．
    ///   ただ，今後どのようなシミュレータを追加するか現状では何とも言えなく，
    ///   どのようにショット速度を逆算すべきかも何とも言えないので，現状そのような関数はライブラリでは提供していません）
    /// - 関数内でシミュレータFCV1を使用して1ショット分シミュレーションを行っています．ですので，この関数の実行は高速とは言えません．
    /// - この関数は解析的にもとめたものでなく，シミュレーション結果から回帰分析で求めた関数です．したがって，特に飛距離にはある程度誤差が存在します．
    ///
    /// \param target_position 目標地点
    ///
    /// \param target_speed 目標地点到達時の速度．
    ///     0 にすればドローショット(石を停止させるショット)，0 より大きくすればヒットショット(石を他の石にぶつけるショット)になる．
    ///
    /// \param rotation ショットの回転方向
    ///
    /// \return 推測されたストーンの初速ベクトル
    dc::Vector2 EstimateShotVelocityFCV1(dc::Vector2 const &target_position, float target_speed, dc::moves::Shot::Rotation rotation)
    {
        assert(target_speed >= 0.f);
        assert(target_speed <= 4.f);

        // 初速度の大きさを逆算する
        // 逆算には専用の関数を用いる．

        float const v0_speed = [&target_position, target_speed]
        {
            auto const target_r = target_position.Length();
            assert(target_r > 0.f);

            if (target_speed <= 0.05f)
            {
                float constexpr kC0[] = {0.0005048122574925176, 0.2756242531609261};
                float constexpr kC1[] = {0.00046669575066030805, -29.898958358378636, -0.0014030973174948508};
                float constexpr kC2[] = {0.13968687866736632, 0.41120940058777616};

                float const c0 = kC0[0] * target_r + kC0[1];
                float const c1 = -kC1[0] * std::log(target_r + kC1[1]) + kC1[2];
                float const c2 = kC2[0] * target_r + kC2[1];

                return std::sqrt(c0 * target_speed * target_speed + c1 * target_speed + c2);
            }
            else if (target_speed <= 1.f)
            {
                float constexpr kC0[] = {-0.0014309170115803444, 0.9858457898438147};
                float constexpr kC1[] = {-0.0008339331735471273, -29.86751291726946, -0.19811799977982522};
                float constexpr kC2[] = {0.13967323742978, 0.42816312110477517};

                float const c0 = kC0[0] * target_r + kC0[1];
                float const c1 = -kC1[0] * std::log(target_r + kC1[1]) + kC1[2];
                float const c2 = kC2[0] * target_r + kC2[1];

                return std::sqrt(c0 * target_speed * target_speed + c1 * target_speed + c2);
            }
            else
            {
                float constexpr kC0[] = {1.0833113118071224e-06, -0.00012132851917870833, 0.004578093297561233, 0.9767006869364527};
                float constexpr kC1[] = {0.07950648211492622, -8.228225657195706, -0.05601306077702578};
                float constexpr kC2[] = {0.14140440186382008, 0.3875782508767419};

                float const c0 = kC0[0] * target_r * target_r * target_r + kC0[1] * target_r * target_r + kC0[2] * target_r + kC0[3];
                float const c1 = -kC1[0] * std::log(target_r + kC1[1]) + kC1[2];
                float const c2 = kC2[0] * target_r + kC2[1];

                return std::sqrt(c0 * target_speed * target_speed + c1 * target_speed + c2);
            }
        }();

        assert(target_speed < v0_speed);

        // 一度シミュレーションを行い，発射方向を決定する

        dc::Vector2 const delta = [rotation, v0_speed, target_speed]
        {
            float const rotation_factor = rotation == dc::moves::Shot::Rotation::kCCW ? 1.f : -1.f;

            // シミュレータは FCV1 シミュレータを使用する．
            thread_local std::unique_ptr<dc::ISimulator> s_simulator;
            if (s_simulator == nullptr)
            {
                s_simulator = dc::simulators::SimulatorFCV1Factory().CreateSimulator();
            }

            dc::ISimulator::AllStones init_stones;
            init_stones[0].emplace(dc::Vector2(), 0.f, dc::Vector2(0.f, v0_speed), 1.57f * rotation_factor);
            s_simulator->SetStones(init_stones);

            while (!s_simulator->AreAllStonesStopped())
            {
                auto const &stones = s_simulator->GetStones();
                auto const speed = stones[0]->linear_velocity.Length();
                if (speed <= target_speed)
                {
                    return stones[0]->position;
                }
                s_simulator->Step();
            }

            return s_simulator->GetStones()[0]->position;
        }();

        float const delta_angle = std::atan2(delta.x, delta.y); // 注: delta.x, delta.y の順番で良い
        float const target_angle = std::atan2(target_position.y, target_position.x);
        float const v0_angle = target_angle + delta_angle; // 発射方向

        return dc::Vector2(v0_speed * std::cos(v0_angle), v0_speed * std::sin(v0_angle));
    }

    // グローバル変数

    dc::Team g_team; /// 自分のチーム
    dc::GameSetting g_game_setting;
    std::unique_ptr<dc::ISimulator> g_simulator;
    std::unique_ptr<dc::ISimulatorStorage> g_simulator_storage;
    std::array<std::unique_ptr<dc::IPlayer>, 4> g_players;

}  
int main(int argc, char const *argv[])
{
    using boost::asio::ip::tcp;
    using nlohmann::json;
    using ShotRotation = dc::moves::Shot::Rotation;
    // 15*stone:x,y,team(-1,0,1)
    // shot(x,y,rot(-1,1))
    boost::asio::io_context io_context;
    // Server server(io_context)
    // server.start_accept();
    tcp::acceptor acc(io_context, tcp::endpoint(tcp::v4(), 10000+atoi(argv[1])));
    tcp::socket socket(io_context);
    acc.accept(socket);

    clock_t start = clock();

    // boost::asio::write(socket, boost::asio::buffer("OK"));
    std::cout << "connect" << std::endl;

    
    while (true)
    {
        boost::asio::streambuf receive_buffer;
        boost::system::error_code error;
        boost::asio::read_until(socket, receive_buffer, '\n', error);
        std::string stone_data;
        const char* data;
        if (error && error != boost::asio::error::eof)
        {
            std::cout << "receive failed: " << error.message() << std::endl;
            //break;
        }
        else
        {
            //data = boost::asio::buffer_cast<const char *>(receive_buffer.data());
            std::istream is(&receive_buffer);
            std::getline(is,stone_data);
        }
        
        //std::string stone_data = data;
        
        json j_stone = json::parse(stone_data);
        
        dc::ISimulator::AllStones allstones;
        
        for (auto i = 0; i < 15; i++)
        {
            std::string s = "stone" + std::to_string(i + 1);
            //std::cout << j_stone[s] << std::endl;
            if (j_stone[s][2] != 0)
            {
                allstones[i].emplace(dc::Vector2(j_stone[s][0], j_stone[s][1]), 0.f, dc::Vector2(), 0);
                //std::cout << j_stone[s][0]<<j_stone[s][1] << std::endl;
            }
            else
            {
                break;
            }
        }
        std::string s = "shot";
        dc::Vector2 shot;
        double  y =(double)j_stone[s][1];
        double shot_pos_y=(y<0)?-y:y;
        double shot_pos_x=(double)j_stone[s][0];
        
        auto shot_speed=(y<0)?3.0:0.0;
        if((int)j_stone[s][2]==1){
            shot=EstimateShotVelocityFCV1(dc::Vector2::Vector2(shot_pos_x,shot_pos_y),shot_speed, ShotRotation::kCCW);
        }else{
            shot=EstimateShotVelocityFCV1(dc::Vector2::Vector2(shot_pos_x,shot_pos_y), shot_speed, ShotRotation::kCW);
        }
        allstones[15].emplace(dc::Vector2(), 0.f, shot, 1.57f * (double)j_stone[s][2]);
        thread_local std::unique_ptr<dc::ISimulator> s_simulator;
        if (s_simulator == nullptr)
        {
            s_simulator = dc::simulators::SimulatorFCV1Factory().CreateSimulator();
        }
        s_simulator->SetStones(allstones);
        while (!s_simulator->AreAllStonesStopped())
        {
            s_simulator->Step();
        }
        auto const &after_allstones = s_simulator->GetStones();
        for (auto i = 0; i < 16; i++)
        {
            std::string s = "stone" + std::to_string(i + 1);
            auto pos = after_allstones[i]->position;
            //std::cout<<pos.x<<","<<pos.y<<std::endl;
            j_stone[s][0] = pos.x;
            j_stone[s][1] = pos.y;
        }
        j_stone["stone16"][2] = 1;
        j_stone["shot"][0]=shot.x;
        j_stone["shot"][1]=shot.y;
        auto const output_message = j_stone.dump();
        //std::cout << output_message << std::endl;
        boost::asio::write(socket, boost::asio::buffer(output_message), error);
    }

    clock_t end = clock();
    std::cout << "duration = " << (double)(end - start) / CLOCKS_PER_SEC << "sec.\n";
    return 0;
}