import json
import pathlib
import torch
import random
import click
import numpy as np
from typing import List
import os

from nn.network.dual_net import DualNet
from dc3client import SocketClient
from dc3client.models import StoneRotation, Position, Coordinate, Stones
from nn.feature import generate_input_planes
from nn.utility import get_torch_device, load_network
from board.constant import BOARD_SIZE, VX_MIN, VX_MAX, VY_MIN, VY_MAX
from other.other import generate_move_from_policy

from mcts.curling_env import CurlingEnv
from mcts.sh_player import Player

from other.score_test import score_test

from mcts.FCV1Simulation.src.build.Release.simulator import StoneSimulator

current_dir = os.path.dirname(__file__)
model_path = os.path.join(current_dir, "model\sl-model.bin")

def convert_stones_to_list(stones: Stones) -> List[dict]:
    result = [None] * 16  # 16要素のリストを作成し、全てをNoneで初期化
    for i, coordinate in enumerate(stones.team0):
        if coordinate.angle is not None and coordinate.position[0].x is not None and coordinate.position[0].y is not None:
            data = {
                "angle": coordinate.angle,
                "angular_velocity": 0.0,
                "linear_velocity": {"x": 0.0, "y": 0.0},
                "position": {"x": coordinate.position[0].x, "y": coordinate.position[0].y}
            }
            result[i] = data

    for i, coordinate in enumerate(stones.team1):
        if coordinate.angle is not None and coordinate.position[0].x is not None and coordinate.position[0].y is not None:
            data = {
                "angle": coordinate.angle,
                "angular_velocity": 0.0,
                "linear_velocity": {"x": 0.0, "y": 0.0},
                "position": {"x": coordinate.position[0].x, "y": coordinate.position[0].y}
            }
            result[i + 8] = data

    return result

def convert_stones_to_list_mcts(stones: Stones) -> List[dict]:
    result = [None] * 16  # 16要素のリストを作成し、全てをNoneで初期化

    for i, coordinate in enumerate(stones.team0):
        if coordinate.angle is not None and coordinate.position[0].x is not None and coordinate.position[0].y is not None:
            data = [coordinate.angle, coordinate.position[0].x, coordinate.position[0].y]
            result[i] = data

    for i, coordinate in enumerate(stones.team1):
        if coordinate.angle is not None and coordinate.position[0].x is not None and coordinate.position[0].y is not None:
            data = [coordinate.angle, coordinate.position[0].x, coordinate.position[0].y]
            result[i + 8] = data

    return result
    



@click.command()
@click.option('--port', type=int, default=10001, help='Port number (default: 10000)')

def main(**kwargs):
    # 機械学習のモデルなど、時間のかかる処理はここで行います。
    # 通信プロトコルの解説において、is_readyを受け取ってからreadyを返すまでに行うことを推奨しています。
    # しかしながら、そのタイミングでエラーが生じるとサーバー自体の動作が停止してしまうため、すべての準備が終わってから接続を行うことを推奨します。
    # dc3の推奨に従う場合、以下のようになります。
    # 1. auto_startをFalseにしてSocketClientを初期化する
    #   cli = SocketClient(auto_start=False)
    # 2. サーバーに接続する
    #   cli.connect(cli.server)
    # 3. dcを受け取る
    #   cli.dc_receive()
    # 4. dc_okを送信する
    #   cli.dc_ok()
    # 5. is_readyを受け取る
    #   cli.is_ready_recv()
    # 6. モデルを読み込むなどの準備を行う
    # 7. ready_okを送信する
    #   cli.ready_ok()
    # 8. サーバーからの開始指示を待つ
    #   cli.get_new_game()
    port = kwargs['port']
    print("1")

    # SocketClientには以下の引数を渡すことができます
    # host : デジタルカーリングを実行しているサーバーのIPアドレスを指定します。名前解決可能であればホスト名でも指定可能です。
    # port : デジタルカーリングを実行しているサーバーの指定されたポート番号を指定します。
    # client_name : クライアントの名前を指定します。デフォルトでは"AI0"となっています。
    # auto_start : サーバーに接続した際に自動で試合を開始するかどうかを指定します。デフォルトではTrueとなっています。
    # これは、dc3のコンバート機能のみを使用したいときにサーバーを起動する必要をなくすために用意されています。
    # rate_limit : 通信のレート制限を指定します。デフォルトでは2.0秒に1回となっています。早すぎるとサーバーから切断される可能性があります。192.168.11.55
    cli = SocketClient(host="localhost", port=port, client_name="KuraCurling_SHOT", auto_start=True, rate_limit=0.5)
    #cli = SocketClient(host="digitalcurling.ddns.net", port=port, client_name="SAMPLE_AI0", auto_start=True, rate_limit=1.0)
    print("2")

    # ログを出力するディレクトリを指定します。デフォルトでは"logs/"となっています。
    #log_dir = pathlib.Path("logs")

    # データ保存時に、軌跡データを削除するかどうかを指定します。デフォルトではTrueとなっています。
    # 軌跡データを保存すると容量が膨大になるため、必要ない場合はTrueにしてください。
    remove_trajectory = True

    # 自分がteam0かteam1かを取得します
    my_team = cli.get_my_team()
    cli.logger.info(f"my_team :{my_team}")
    print("3")

    # dcやis_readyをdataclass形式で取得し、保存しやすいようにdict形式に変換します。
    dc = cli.get_dc()
    dc_message = cli.convert_dc(dc)
    is_ready = cli.get_is_ready()

    device = get_torch_device(use_gpu=False)
    network = load_network(model_path, use_gpu=False)
    network.to(device)
    
    simple_player = Player(
        ucb_const = 0.03,
        pw_const = 0.1,
        init_temperature = 1.0,
        out_temperature = 0.0,
        num_init = 100,
        num_sample = 3,
        l = 1.0,
        max_depth = 2,
        stone_simulator=StoneSimulator()
    )

    is_ready_message = cli.convert_is_ready(is_ready)

    # 試合を開始します
    while True:

        # updateを受け取ります
        cli.update()

        # 試合状況を取得します
        # 現在の情報は、match_data.update_listに順番に格納されています
        match_data = cli.get_match_data()
        #print(match_data.update_list[-1].state.stones.team0)
        
        

        # winnerが存在するかどうかで、試合が終了しているかどうかを確認します
        if (winner := cli.get_winner()) is not None:
            # game end
            if my_team == winner:
                # 勝利
                cli.logger.info("WIN")
            else:
                # 敗北
                cli.logger.info("LOSE")
            # 試合が終了したらループを抜けます
            break

        # 次のチームが自分のチームかどうかを確認します
        next_team = cli.get_next_team()

        # 次のチームが自分のチームであれば、moveを送信します
        if my_team == next_team:
            # 実際の投球は、move関数を呼び出すことで行います
            # move関数の引数は、x, y, rotationの3つです
            # x, yはそれぞれ投球する石のx(横)方向成分、y(縦)方向成分を指定します
            # rotationは投球する石の回転方向を指定します
            # このとき、rotationにはStoneRotationクラスの値を指定します
            # StoneRotation.clockwise : 時計回り
            # StoneRotation.inturn : インターン = 時計回り
            # StoneRotation.counterclockwise : 反時計回り
            # StoneRotation.outturn : アウトターン = 反時計回り
            #rint("team: ", type(my_team))
            print(match_data.update_list[-1].state.thinking_time_remaining)
            #-0.121424 2.30874 cw
            if match_data.update_list[-1].state.shot == 0:
                cli.move(x=-0.121424, y=2.30874, rotation=StoneRotation.clockwise)
            elif match_data.update_list[-1].state.shot < 14:
                stones = convert_stones_to_list(match_data.update_list[-1].state.stones)
                inputplanes = generate_input_planes(stones, match_data.update_list[-1].state.shot)
                selected_x, selected_y, selected_rotation = generate_move_from_policy(network, inputplanes)
                cli.move(x=selected_x, y=selected_y, rotation=selected_rotation)
            elif match_data.update_list[-1].state.shot == 15:
                print("MCTS!")
                stones = convert_stones_to_list_mcts(match_data.update_list[-1].state.stones)
                env = CurlingEnv(num_shot = match_data.update_list[-1].state.shot, stones = stones, team=my_team)
                
                score_test(stones)

                selected_xys, _ = simple_player.think(root_env=env, num_playout=1023, num_init=512)
                cli.move(x=selected_xys[0], y=selected_xys[1], rotation=selected_xys[2])
            elif match_data.update_list[-1].state.shot == 14:
                print("15MCTS!")
                stones = convert_stones_to_list_mcts(match_data.update_list[-1].state.stones)
                env = CurlingEnv(num_shot = match_data.update_list[-1].state.shot, stones = stones, team=my_team)
                
                score_test(stones)

                selected_xys, _ = simple_player.think(root_env=env, num_playout=511, num_init=256)
                cli.move(x=selected_xys[0], y=selected_xys[1], rotation=selected_xys[2])
                #stones = convert_stones_to_list(match_data.update_list[-1].state.stones)
                #inputplanes = generate_input_planes(stones, match_data.update_list[-1].state.shot)
                #selected_x, selected_y, selected_rotation = akirame(network, inputplanes, stones, my_team, simple_player, num_node=9.0, max_time=15.0)
                #cli.move(x=selected_x, y=selected_y, rotation=selected_rotation)
            
            """
            if match_data.update_list[-1].state.shot == 15:
                print("MCTS!")
                stones = convert_stones_to_list_mcts(match_data.update_list[-1].state.stones)
                env = CurlingEnv(num_shot = match_data.update_list[-1].state.shot, stones = stones, team=my_team)
                selected_xys, _ = simple_player.think(root_env=env, num_playout=1000, max_time = 10.0)
                cli.move(x=selected_xys[0], y=selected_xys[1], rotation=selected_xys[2])
            elif match_data.update_list[-1].state.shot == 14:
                print("15MCTS!")
                stones = convert_stones_to_list(match_data.update_list[-1].state.stones)
                inputplanes = generate_input_planes(stones, match_data.update_list[-1].state.shot)
                selected_x, selected_y, selected_rotation = akirame(network, inputplanes, stones, my_team, simple_player)
                cli.move(x=selected_x, y=selected_y, rotation=selected_rotation)
            else:
                stones = convert_stones_to_list(match_data.update_list[-1].state.stones)
                inputplanes = generate_input_planes(stones, match_data.update_list[-1].state.shot)
                selected_x, selected_y, selected_rotation = generate_move_from_policy(network, inputplanes)
                cli.move(x=selected_x, y=selected_y, rotation=selected_rotation)"""
            print(match_data.update_list[-1].state.thinking_time_remaining)
        else:
            # 次のチームが自分のチームでなければ、何もしません
            continue

    # 試合が終了したら、clientから試合データを取得します
    move_info = cli.get_move_info()
    update_list, trajectory_list = cli.get_update_and_trajectory(remove_trajectory)

    '''# 試合データを保存します、
    update_dict = {}

    for update in update_list:
        # updateをdict形式に変換します
        update_dict = cli.convert_update(update, remove_trajectory)

    # updateを保存します、どのように保存するかは任意です
    with open("data.json", "w", encoding="UTF-8") as f:
        json.dump(update_dict, f, indent=4)'''


if __name__ == '__main__':
    main()
